#!/usr/bin/env python3
# Copyright (c) Microsoft. All rights reserved.

import mssqlconfhelper
import mssqlsettings

_ = mssqlconfhelper._

#
# Supported settings list
#
supportedSettingsList = []
supportedSectionsList = []


def getSupportedSectionsList():
    """Get supported sections list

    Returns:
        Supported sections list
    """

    return supportedSectionsList


def findSetting(section_name, setting_name=None):
    """Find a setting

    Args:
        section_name (str): Name of the section.
                            For legacy purposes section_name can include setting_name as well
                            For example: section_name.setting_name
        setting_name (str): Name of the setting. Optional if section_name contains setting_name as well.


    Returns:
        Setting information
    """

    result = None

    if not setting_name:
        split = section_name.split('.', 1)
        if len(split) > 1:
            section_name, setting_name = split
        else:
            return None

    for setting in supportedSettingsList:
        if setting.section == section_name and (setting.name == setting_name or setting.sectionOnly == True):
            result = setting
    if result is not None:
        matches = [result]
    else:
        matches = []

    if len(matches) == 0:
        return None
    else:
        return matches[0]


def addTraceFlag(config, traceflag):
    """Add a trace flag to config

    Args:
        config (class): Configuration parser object
        traceflag (str): Traceflag to add
    """

    traceFlagDict = {}

    if(config.has_section(mssqlsettings.SectionForSetting.traceflag) == True):
        traceFlagDict = dict(config.items(mssqlsettings.SectionForSetting.traceflag))

        if (str(traceflag) in list(traceFlagDict.values())):
            return

    config.remove_section(mssqlsettings.SectionForSetting.traceflag)
    count = 0
    config.add_section(mssqlsettings.SectionForSetting.traceflag)

    for key in traceFlagDict:
        traceFlagName = "traceflag" + str(count)
        count += 1
        config.set(mssqlsettings.SectionForSetting.traceflag, traceFlagName, traceFlagDict[key])

    config.set(mssqlsettings.SectionForSetting.traceflag, ("traceflag" + str(count)), str(traceflag))
    mssqlconfhelper.printRestartRequiredMessage()


def removeTraceFlag(config, traceflag):
    """Remove trace flag

    Args:
        config (class); Configuration parser object
        traceflag (str): Traceflag
    """

    if not config.has_section(mssqlsettings.SectionForSetting.traceflag):
        return

    traceflag = str(traceflag)
    traceFlagDict = dict(config.items(mssqlsettings.SectionForSetting.traceflag))
    found = False

    for key in traceFlagDict:
        if(traceFlagDict[key] == traceflag):
            config.remove_option(mssqlsettings.SectionForSetting.traceflag, key)
            mssqlconfhelper.printRestartRequiredMessage()
            found = True

    if(found == False):
        return


def setSetting(config, setting, settingValue, suppressRestartMessage=False):
    """Set a configuration setting

    Args:
        config (class); Configuration parser object
        setting (setting): Setting name
        settingValue (str): Setting value
    """

    # Validate the setting value for the chosen setting
    if not setting.validateSetting(settingValue):
        return False

    if not config.has_section(setting.section):
        config.add_section(setting.section)

    config.set(setting.section, setting.name, settingValue)

    if(setting.restart_required and not suppressRestartMessage):
        mssqlconfhelper.printRestartRequiredMessage()

    return True


def unsetSetting(config, setting):
    """Unset a configuration setting

    Args:
        config (class); Configuration parser object
        setting (setting): Setting name
    """

    if (config.has_option(setting.section, setting.name)):
        config.remove_option(setting.section, setting.name)
        return True

    return False


def unsetSectionKeyVal(config, setting, key):
    """Unset a configuration setting for sections
        without settings.

    Args:
        config (class); Configuration parser object
        setting (setting): instance of setting
        key     (key): left side of setting value pair
    """

    if (config.has_option(setting.section, key)):
        config.remove_option(setting.section, key)
        return True

    return False


def unsetSectionOnlySettings(config, setting):
    """Unset a configuration setting for sections
        without settings. Unsets all the existing
        values.

    Args:
        config (class); Configuration parser object
        setting (setting): instance of setting
    """

    settingsDictionary = mssqlconfhelper.getSettings(mssqlconfhelper.configurationFilePath,
                                                     setting.section)
    if len(settingsDictionary):
        for key in settingsDictionary:
            unsetSectionKeyVal(config, setting, key)

        return True
    else:
        return False


def validateConfig(config):
    """Validate configuration

    Args:
        config (class): Configuration parser object
    """

    validationSuccessful = True

    sectionsInConfigFile = config.sections()
    for section in sectionsInConfigFile:
        if not section in supportedSectionsList:
            validationSuccessful = False
            print((_("WARNING: Unknown configuration section '%s'.") % section))

    # Validate the settings in all supported sections
    for section in config.sections():
        settingsInSection = config.items(section)
        for setting in settingsInSection:
            setting_name = setting[0]
            settingValue = setting[1]

            # Handle trace flags setting
            if (section == mssqlsettings.SectionForSetting.traceflag):
                if not findSetting("traceflag.traceflag").validateSetting(settingValue):
                    validationSuccessful = False
                    print((_("WARNING: Invalid traceflag '%s'.") % setting_name))

            # Handle uncmapping section
            elif(section == mssqlsettings.SectionForSetting.uncmapping):
                if not findSetting("uncmapping.").validateSetting(settingValue, False, setting_name):
                    validationSuccessful = False
                    print(_("WARNING: Invalid UNC mapping '%s'='%s'.") % (setting_name, settingValue))

            # All other settings
            else:
                setting = findSetting("%s.%s" % (section, setting_name))

                if (setting == None):
                    validationSuccessful = False
                    print(_("WARNING: Unknown setting '%s.%s'.") % (section, setting_name))
                else:
                    if not setting.validateSetting(config.get(section, setting_name), validate_verb_used=True):
                        validationSuccessful = False
                        print(_("WARNING: Setting '%s.%s' cannot be validated.") % (section, setting_name))

    return validationSuccessful


def setSectionOnlySettings(config, setting, value, suppressRestartMessage = False):
    """Set Setting for Sections without specific
        setting name. This is used during setup
        to use the value from environment variable
        and set it. Environment variable can contain
        multiple entries separated by ":" so this function
        splits it and then calls setSectionKeyVal on each one
        of them.

    Args:
        config (class): Configuration parser object
        setting (setting): instance of mssqlregistrysetting
        value (str): assumes left and right part are separated
        by '=' and multiple settings are separated by ':'

    """

    values = value.split(":")
    for item in values:
        key,val = item.split("=")
        if not setSectionKeyVal(config, setting, key, val, suppressRestartMessage):
            return False
    return True


def setSectionKeyVal(config, chosenSection, key, val, suppressRestartMessage = False):
    """Set a configuration setting for a section

    Args:
        config (class); Configuration parser object
        chosenSetting (setting): instance of MssqlRegistrySetting
        key (str):  left side or Setting name
        val (str): right side or value of the setting
    """

    # Validate the setting value for the chosen setting
    if not chosenSection.validateSetting(val, False, key):
        return False

    if not config.has_section(chosenSection.section):
        config.add_section(chosenSection.section)

    config.set(chosenSection.section, key, val)

    if (chosenSection.restart_required and not suppressRestartMessage):
        mssqlconfhelper.printRestartRequiredMessage()

    return True


def initialize():
    """Initialize settings manager
    """

    global supportedSectionsList
    global supportedSettingsList

    supportedSettingsList.append(mssqlsettings.TcpPortSetting("tcpport",
        "MSSQL_TCP_PORT",
        mssqlsettings.SettingValueType.integer,
        _("TCP port for incoming connections"),
        mssqlsettings.SectionForSetting.network,
        True))

    supportedSettingsList.append(mssqlsettings.TcpPortSetting("rpcport",
        "MSSQL_RPC_PORT",
        mssqlsettings.SettingValueType.integer,
        _("TCP port for Rpc endpoint mapper"),
        mssqlsettings.SectionForSetting.network,
        True))

    supportedSettingsList.append(mssqlsettings.IpAddressSetting("ipaddress",
        "MSSQL_IP_ADDRESS",
        mssqlsettings.SettingValueType.string,
        _("IP address for incoming connections"),
        mssqlsettings.SectionForSetting.network,
        True))

    supportedSettingsList.append(mssqlsettings.DirectorySetting("defaultbackupdir",
        "MSSQL_BACKUP_DIR",
        mssqlsettings.SettingValueType.dirPath,
        _("Default directory for backup files"),
        mssqlsettings.SectionForSetting.file_location,
        True))

    supportedSettingsList.append(mssqlsettings.DirectorySetting("defaultdumpdir",
        "MSSQL_DUMP_DIR",
        mssqlsettings.SettingValueType.dirPath,
        _("Default directory for crash dump files"),
        mssqlsettings.SectionForSetting.file_location,
        False))

    supportedSettingsList.append(mssqlsettings.TraceFlagSetting("traceflag",
        None,
        mssqlsettings.SettingValueType.traceFlag,
        _("Trace flag settings"),
        mssqlsettings.SectionForSetting.traceflag,
        True,
        hidden=True))

    supportedSettingsList.append(mssqlsettings.DirectorySetting("defaultlogdir",
        "MSSQL_LOG_DIR",
        mssqlsettings.SettingValueType.dirPath,
        _("Default directory for log files"),
        mssqlsettings.SectionForSetting.file_location,
        True))

    supportedSettingsList.append(mssqlsettings.DirectorySetting("defaultdatadir",
        "MSSQL_DATA_DIR",
        mssqlsettings.SettingValueType.dirPath,
        _("Default directory for data files"),
        mssqlsettings.SectionForSetting.file_location,
        True))

    supportedSettingsList.append(mssqlsettings.NumErrorLogsSetting("numerrorlogs",
        "MSSQL_NUMBER_ERROR_LOGS",
        mssqlsettings.SettingValueType.integer,
        _("Number of error log maintained before cycling the log."),
        mssqlsettings.SectionForSetting.errorlog,
        True))

    supportedSettingsList.append(mssqlsettings.BooleanSetting("hadrenabled",
        "MSSQL_ENABLE_HADR",
        mssqlsettings.SettingValueType.boolean,
        _("Allow SQL Server to use availability groups for high availability and disaster recovery"),
        mssqlsettings.SectionForSetting.hadr,
        True,
        "1", # true value
        "0")) # false value

    supportedSettingsList.append(mssqlsettings.CoreDumpTypeSetting("coredumptype",
        "MSSQL_CORE_DUMP_TYPE",
        mssqlsettings.SettingValueType.string,
        _("Core dump type to capture: mini, miniplus, filtered, full"),
        mssqlsettings.SectionForSetting.coredump,
        False))

    supportedSettingsList.append(mssqlsettings.BooleanSetting("captureminiandfull",
        "MSSQL_CORE_CAPTURE_MINIANDFULL",
        mssqlsettings.SettingValueType.boolean,
        _("Capture both mini and full core dumps"),
        mssqlsettings.SectionForSetting.coredump,
        False,
        "true",  # true value
        "false")) # false value

    supportedSettingsList.append(mssqlsettings.BooleanSetting("disablecoredump",
        "MSSQL_DISABLE_COREDUMP",
        mssqlsettings.SettingValueType.boolean,
        _("SQL Server disable core dump"),
        mssqlsettings.SectionForSetting.coredump,
        True,
        "true",  # true value 1
        "false")) # false value 0

    supportedSettingsList.append(mssqlsettings.BooleanSetting("forceencryption",
        "MSSQL_FORCE_ENCRYPTION",
        mssqlsettings.SettingValueType.boolean,
        _("Force encryption of incoming client connections"),
        mssqlsettings.SectionForSetting.network,
        True,
        "1", # true value
        "0")) # false value

    supportedSettingsList.append(mssqlsettings.TlsCertificateSetting("tlscert",
        "MSSQL_TLS_CERTIFICATE_FILE",
        mssqlsettings.SettingValueType.string,
        _("Path to certificate file for encrypting incoming client connections"),
        mssqlsettings.SectionForSetting.network,
        True))

    supportedSettingsList.append(mssqlsettings.TlsKeySetting("tlskey",
        "MSSQL_TLS_KEY_FILE",
        mssqlsettings.SettingValueType.string,
        _("Path to private key file for encrypting incoming client connections"),
        mssqlsettings.SectionForSetting.network,
        True))

    supportedSettingsList.append(mssqlsettings.TlsProtocolsSetting("tlsprotocols",
        "MSSQL_TLS_PROTOCOLS",
        mssqlsettings.SettingValueType.string,
        _("TLS protocol versions allowed for encrypted incoming client connections"),
        mssqlsettings.SectionForSetting.network,
        True))

    supportedSettingsList.append(mssqlsettings.TlsCiphersSetting("tlsciphers",
        "MSSQL_TLS_CIPHERS",
        mssqlsettings.SettingValueType.string,
        _("TLS ciphers allowed for encrypted incoming client connections"),
        mssqlsettings.SectionForSetting.network,
        True))

    supportedSettingsList.append(mssqlsettings.LcidSetting(mssqlconfhelper.lcid,
        "MSSQL_LCID",
        mssqlsettings.SettingValueType.string,
        _("Locale identifier for SQL Server to use (e.g. 1033 for US - English)"),
        mssqlsettings.SectionForSetting.language,
        True))

    supportedSettingsList.append(mssqlsettings.AgentErrorLoggingLevelSetting("errorlogginglevel",
        "MSSQL_AGENT_LOGGING_LEVEL",
        mssqlsettings.SettingValueType.integer,
        _("SQL Agent logging level bitmask - 1=Errors, 2=Warnings, 4=Info"),
        mssqlsettings.SectionForSetting.sqlagent,
        True))

    supportedSettingsList.append(mssqlsettings.IntegerSetting("logginglevel",
        "MSSQL_CONNECTOR_LOGGING_LEVEL",
        mssqlsettings.SettingValueType.integer,
        _("SQL Connector logging level - 0=Info(default), 1=Error, 2=No log"),
        mssqlsettings.SectionForSetting.sqlconnector,
        False))

    supportedSettingsList.append(mssqlsettings.AgentErrorLogFileSetting("errorlogfile",
        "MSSQL_AGENT_LOG_FILE",
        mssqlsettings.SettingValueType.string,
        _("SQL Agent log file path"),
        mssqlsettings.SectionForSetting.sqlagent,
        True))

    supportedSettingsList.append(mssqlsettings.BooleanSetting("enabled",
        "MSSQL_AGENT_ENABLED",
        mssqlsettings.SettingValueType.boolean,
        _("Enable or disable SQLAgent"),
        mssqlsettings.SectionForSetting.sqlagent,
        True,
        "true",  # true value
        "false")) # false value

    supportedSettingsList.append(mssqlsettings.BooleanSetting("startupwaitforalldb",
        "MSSQL_AGENT_STARTUP_WAIT_FOR_ALL_DB",
        mssqlsettings.SettingValueType.boolean,
        _("Set to 1 (default) if SqlAgent should wait for all databases on startup; set to 0 to wait for MSDB only"),
        mssqlsettings.SectionForSetting.sqlagent,
        False,
        "1", # true value
        "0")) # false value

    supportedSettingsList.append(mssqlsettings.IntegerSetting("jobhistorymaxrows",
        "MSSQL_AGENT_JOB_HISTORY_MAX_ROWS",
        mssqlsettings.SettingValueType.integer,
        _("Maximum job history log size (in rows)"),
        mssqlsettings.SectionForSetting.sqlagent,
        False))

    supportedSettingsList.append(mssqlsettings.IntegerSetting("jobhistorymaxrowsperjob",
        "MSSQL_AGENT_JOB_HISTORY_MAX_ROWS_PER_JOB",
        mssqlsettings.SettingValueType.integer,
        _("Maximum job history rows per job"),
        mssqlsettings.SectionForSetting.sqlagent,
        False))

    supportedSettingsList.append(mssqlsettings.ExternalPolicyBasedAuthorizationSetting("externalpolicybasedauthorization",
        "MSSQL_GOVERNANCE_EXTERNAL_POLICY_BASED_AUTHORIZATION",
        mssqlsettings.SettingValueType.string,
        _("Indicates whether external policy based authorization is enabled for the server by setting Azure Purview = enabled"),
        mssqlsettings.SectionForSetting.governance,
        True))

    supportedSettingsList.append(mssqlsettings.PurviewPdsEndpointSetting("purviewpdsendpoint",
        "MSSQL_GOVERNANCE_PURVIEW_PDS_ENDPOINT",
        mssqlsettings.SettingValueType.string,
        _("Purview gateway endpoint for retrieving policies"),
        mssqlsettings.SectionForSetting.governance,
        True))

    supportedSettingsList.append(mssqlsettings.PurviewPolicyApiVersionSetting("purviewpolicyapiversion",
        "MSSQL_GOVERNANCE_PURVIEW_POLICY_API_VERSION",
        mssqlsettings.SettingValueType.string,
        _("API version for the Purview policy"),
        mssqlsettings.SectionForSetting.governance,
        True))

    supportedSettingsList.append(mssqlsettings.PurviewResourceEndpointSetting("purviewresourceendpoint",
        "MSSQL_GOVERNANCE_PURVIEW_RESOURCE_ENDPOINT",
        mssqlsettings.SettingValueType.string,
        _("Purview resource endpoint"),
        mssqlsettings.SectionForSetting.governance,
        True))

    supportedSettingsList.append(mssqlsettings.PurviewConfigSettingsSetting("purviewconfigsettings",
        "MSSQL_GOVERNANCE_PURVIEW_CONFIG_SETTINGS",
        mssqlsettings.SettingValueType.string,
        _("Default configuration settings for Purview"),
        mssqlsettings.SectionForSetting.governance,
        True))

    supportedSettingsList.append(mssqlsettings.KerberosKeytabFileSetting("kerberoskeytabfile",
        "MSSQL_KERBEROS_KEYTAB_FILE",
        mssqlsettings.SettingValueType.string,
        _("Kerberos keytab file location"),
        mssqlsettings.SectionForSetting.network,
        True))

    supportedSettingsList.append(mssqlsettings.BooleanSetting("forcesecureldap",
        "MSSQL_FORCE_LDAPS",
        mssqlsettings.SettingValueType.boolean,
        _("Force using LDAPS to contact domain controller"),
        mssqlsettings.SectionForSetting.network,
        True,
        "true",   # true value
        "false")) # false value

    supportedSettingsList.append(mssqlsettings.BooleanSetting("disablesssd",
        "MSSQL_DISABLE_SSSD",
        mssqlsettings.SettingValueType.boolean,
        _("Disable querying SSSD for AD account information and default to LDAP calls"),
        mssqlsettings.SectionForSetting.network,
        True,
        "true",   # true value
        "false")) # false value

    supportedSettingsList.append(mssqlsettings.BooleanSetting("enablekdcfromkrb5conf",
        "MSSQL_ENABLE_KDC_FROM_KRB5CONF",
        mssqlsettings.SettingValueType.boolean,
        _("Enable looking up KDC information from krb5.conf"),
        mssqlsettings.SectionForSetting.network,
        True,
        "true",   # true value
        "false")) # false value

    supportedSettingsList.append(mssqlsettings.ADUserSetting("privilegedadaccount",
        "MSSQL_PRIVILEGED_AD_USER",
        mssqlsettings.SettingValueType.string,
        _("Privileged AD user to use for AD authentication"),
        mssqlsettings.SectionForSetting.network,
        True))

    supportedSettingsList.append(mssqlsettings.UpdateFrequencySetting("kerberoscredupdatefrequency",
        "MSSQL_KERBEROS_CRED_UPDATE_FREQUENCY",
        mssqlsettings.SettingValueType.integer,
        _("Time in seconds between checks for kerberos credentials that need to be updated"),
        mssqlsettings.SectionForSetting.network,
        True))

    supportedSettingsList.append(mssqlsettings.DomainListSetting("trustedexternaldomains",
        "MSSQL_TRUSTED_EXTERNAL_DOMAINS",
        mssqlsettings.SettingValueType.string,
        _("Trusted domains outside the AD forest SQL is joined to"),
        mssqlsettings.SectionForSetting.network,
        True))

    supportedSettingsList.append(mssqlsettings.BooleanSetting("ldaphostcanon",
        "MSSQL_LDAP_HOST_CANON",
        mssqlsettings.SettingValueType.boolean,
        _("Canonicalize hostnames with LDAP"),
        mssqlsettings.SectionForSetting.network,
        True,
        "true",   # true value
        "false")) # false value

    supportedSettingsList.append(mssqlsettings.BooleanSetting("fqdnadusernamesenabled",
        "MSSQL_FQDN_AD_USER_NAMES_ENABLED",
        mssqlsettings.SettingValueType.boolean,
        _("Allow AD usernames with FQDN domains instead of NETBIOS"),
        mssqlsettings.SectionForSetting.network,
        True,
        "true",   # true value
        "false")) # false value

    supportedSettingsList.append(mssqlsettings.AgentDatabaseMailProfileSetting("databasemailprofile",
        "MSSQL_AGENT_EMAIL_PROFILE",
        mssqlsettings.SettingValueType.string,
        _("SQL Agent Database Mail profile name"),
        mssqlsettings.SectionForSetting.sqlagent,
        True))

    supportedSettingsList.append(mssqlsettings.BooleanSetting("memory_optimized",
        "MSSQL_MEMORY_OPTIMIZED",
        mssqlsettings.SettingValueType.boolean,
        _("Enable or disable SQL Server memory optimized features - persistent memory file enlightenment, memory protection"),
        mssqlsettings.SectionForSetting.memory,
        True,
        "true",   # true value
        "false")) # false value

    supportedSettingsList.append(mssqlsettings.BooleanSetting("disablememorypressure",
        "MSSQL_DISABLE_MEMORY_PRESSURE",
        mssqlsettings.SettingValueType.boolean,
        _("SQL Server disable memory pressure"),
        mssqlsettings.SectionForSetting.memory,
        True,
        "1",  # true value
        "0")) # false value

    supportedSettingsList.append(mssqlsettings.MemoryLimitSetting("memorylimitmb",
        "MSSQL_MEMORY_LIMIT_MB",
        mssqlsettings.SettingValueType.integer,
        _("SQL Server memory limit (megabytes)"),
        mssqlsettings.SectionForSetting.memory,
        True))

    supportedSettingsList.append(mssqlsettings.BooleanSetting("enablecontainersharedmemory",
        "MSSQL_ENABLE_CONTAINER_SHARED_MEMORY",
        mssqlsettings.SettingValueType.boolean,
        _("Enable or disable use of shared memory when SQL runs inside a container"),
        mssqlsettings.SectionForSetting.memory,
        True,
        "true",   # true value
        "false")) # false value

    supportedSettingsList.append(mssqlsettings.AcceptEulaSetting(mssqlconfhelper.eulaConfigSetting,
        None,
        mssqlsettings.SettingValueType.string,
        _(""),
        mssqlsettings.SectionForSetting.eula,
        True,
        hidden=True))

    supportedSettingsList.append(mssqlsettings.AcceptEulaMlSetting(mssqlconfhelper.eulaMlConfigSetting,
        None,
        mssqlsettings.SettingValueType.string,
        _("Accept EULA for machine learning services"),
        mssqlsettings.SectionForSetting.eula,
        True,
        hidden=True))

    supportedSettingsList.append(mssqlsettings.BooleanSetting(mssqlconfhelper.telemetryConfigSetting,
        None,
        mssqlsettings.SettingValueType.string,
        _("Telemetry status"),
        mssqlsettings.SectionForSetting.telemetry,
        True,
        "true",  # true value
        "false")) # false value

    supportedSettingsList.append(mssqlsettings.DirectorySetting(mssqlconfhelper.telemetryLocalAuditCacheDirectorySetting,
        None,
        mssqlsettings.SettingValueType.dirPath,
        _("Directory for telemetry local audit cache"),
        mssqlsettings.SectionForSetting.telemetry,
        True))

    supportedSettingsList.append(mssqlsettings.MasterDataFileSetting("masterdatafile",
        "MSSQL_MASTER_DATA_FILE",
        mssqlsettings.SettingValueType.filePath,
        _("Master database data file location"),
        mssqlsettings.SectionForSetting.file_location,
        True))

    supportedSettingsList.append(mssqlsettings.MasterLogFileSetting("masterlogfile",
        "MSSQL_MASTER_LOG_FILE",
        mssqlsettings.SettingValueType.filePath,
        _("Master database log file location"),
        mssqlsettings.SectionForSetting.file_location,
        True))

    supportedSettingsList.append(mssqlsettings.MachineKeyFileSetting("machinekeyfile",
        "MSSQL_MACHINE_KEY_FILE",
        mssqlsettings.SettingValueType.filePath,
        _("Location of machine key file"),
        mssqlsettings.SectionForSetting.file_location,
        True))

    supportedSettingsList.append(mssqlsettings.ErrorLogFileSetting("errorlogfile",
        "MSSQL_ERROR_LOG_FILE",
        mssqlsettings.SettingValueType.filePath,
        _("Error log file location"),
        mssqlsettings.SectionForSetting.file_location,
        True))

    supportedSettingsList.append(mssqlsettings.BooleanSetting("alternatewritethrough",
        "MSSQL_ALTERNATE_WRITE_THROUGH",
        mssqlsettings.SettingValueType.boolean,
        _("Enable optimized write through flush for O_DSYNC requests"),
        mssqlsettings.SectionForSetting.control,
        True,
        "1", # true value
        "0")) # false value

    supportedSettingsList.append(mssqlsettings.BooleanSetting("writethrough",
        "MSSQL_WRITE_THROUGH",
        mssqlsettings.SettingValueType.boolean,
        _("Use O_DSYNC for file flag write through requests"),
        mssqlsettings.SectionForSetting.control,
        True,
        "1", # true value
        "0")) # false value

    supportedSettingsList.append(mssqlsettings.Ipv6DnsRecordsLimitSetting("ipv6dnsrecordslimit",
        "MSSQL_IPV6_DNS_RECORDS_LIMIT",
        mssqlsettings.SettingValueType.integer,
        _("Set a limit on the number of AAAA records read from DNS."),
        mssqlsettings.SectionForSetting.network,
        True))

    supportedSettingsList.append(mssqlsettings.HeStackSizeSetting("hestacksize",
        None,
        mssqlsettings.SettingValueType.integer,
        _("Host extension stack size in KB"),
        mssqlsettings.SectionForSetting.control,
        True))

    supportedSettingsList.append(mssqlsettings.UncMapping("",
        "MSSQL_UNC_MAPPING",
        mssqlsettings.SettingValueType.string,
        _("Maps UNC path to a local path. (e.g. ./mssql-conf set uncmapping //servername/sharename /tmp/folder)"),
        mssqlsettings.SectionForSetting.uncmapping,
        False,
        False,
        True))

    supportedSettingsList.append(mssqlsettings.DistributedTransactionTraceValue("turnoffrpcsecurity",
        "MSSQL_DTC_RPC_SECURITY_OFF",
        mssqlsettings.SettingValueType.integer,
        _("Enable or disable RPC security for distributed transactions"),
        mssqlsettings.SectionForSetting.distributedtransaction,
        True))

    supportedSettingsList.append(mssqlsettings.DistributedTransactionIntegerValue("allowonlysecurerpccalls",
        "MSSQL_DTC_RPC_SECURE_ONLY",
        mssqlsettings.SettingValueType.integer,
        _("Configure secure only rpc calls for distributed transactions"),
        mssqlsettings.SectionForSetting.distributedtransaction,
        True))

    supportedSettingsList.append(mssqlsettings.DistributedTransactionIntegerValue("fallbacktounsecurerpcifnecessary",
        "MSSQL_DTC_RPC_IF_FALLBACK_UNSECURE",
        mssqlsettings.SettingValueType.integer,
        _("Configure security only rpc calls for distributed transactions"),
        mssqlsettings.SectionForSetting.distributedtransaction,
        True))

    supportedSettingsList.append(mssqlsettings.DistributedTransactionIntegerValue("maxlogsize",
        "MSSQL_DTC_MAX_LOG_SIZE",
        mssqlsettings.SettingValueType.integer,
        _("DTC log file size in MB. Default is 64MB"),
        mssqlsettings.SectionForSetting.distributedtransaction,
        True))

    supportedSettingsList.append(mssqlsettings.DistributedTransactionIntegerValue("memorybuffersize",
        "MSSQL_DTC_MEMORY_BUFFER_SIZE",
        mssqlsettings.SettingValueType.integer,
        _("Circular buffer size in which traces are stored. This size is in MB and default is 10MB"),
        mssqlsettings.SectionForSetting.distributedtransaction,
        True))

    supportedSettingsList.append(mssqlsettings.DistributedTransactionTraceValue("trace_misc",
        "MSSQL_DTC_TRACE_MISC",
        mssqlsettings.SettingValueType.integer,
        _("Traces that cannot be categorized into the other categories"),
        mssqlsettings.SectionForSetting.distributedtransaction,
        True))

    supportedSettingsList.append(mssqlsettings.DistributedTransactionTraceValue("trace_cm",
        "MSSQL_DTC_TRACE_CM",
        mssqlsettings.SettingValueType.integer,
        _("Traces in the connection manager"),
        mssqlsettings.SectionForSetting.distributedtransaction,
        True))

    supportedSettingsList.append(mssqlsettings.DistributedTransactionTraceValue("trace_trace",
        "MSSQL_DTC_TRACE_TRACE",
        mssqlsettings.SettingValueType.integer,
        _("The trace infrastructure itself"),
        mssqlsettings.SectionForSetting.distributedtransaction,
        True))

    supportedSettingsList.append(mssqlsettings.DistributedTransactionTraceValue("trace_svc",
        "MSSQL_DTC_TRACE_SVC",
        mssqlsettings.SettingValueType.integer,
        _("Traces service and .exe file startup"),
        mssqlsettings.SectionForSetting.distributedtransaction,
        True))

    supportedSettingsList.append(mssqlsettings.DistributedTransactionTraceValue("trace_gateway",
        "MSSQL_DTC_TRACE_GATEWAY",
        mssqlsettings.SettingValueType.integer,
        _("Traces Gateway source"),
        mssqlsettings.SectionForSetting.distributedtransaction,
        True))

    supportedSettingsList.append(mssqlsettings.DistributedTransactionTraceValue("trace_contact",
        "MSSQL_DTC_TRACE_CONTACT",
        mssqlsettings.SettingValueType.integer,
        _("Traces the contact pool and contacts"),
        mssqlsettings.SectionForSetting.distributedtransaction,
        True))

    supportedSettingsList.append(mssqlsettings.DistributedTransactionTraceValue("trace_util",
        "MSSQL_DTC_TRACE_UTIL",
        mssqlsettings.SettingValueType.integer,
        _("Traces utility routines that are called from multiple locations"),
        mssqlsettings.SectionForSetting.distributedtransaction,
        True))

    supportedSettingsList.append(mssqlsettings.DistributedTransactionTraceValue("trace_xa",
        "MSSQL_DTC_TRACE_XA",
        mssqlsettings.SettingValueType.integer,
        _("XA Transaction Manager (XATM) tracing source"),
        mssqlsettings.SectionForSetting.distributedtransaction,
        True))

    supportedSettingsList.append(mssqlsettings.DistributedTransactionTraceValue("trace_proxy",
        "MSSQL_DTC_TRACE_PROXY",
        mssqlsettings.SettingValueType.integer,
        _("Traces that are generated in the MSDTC proxy"),
        mssqlsettings.SectionForSetting.distributedtransaction,
        True))

    supportedSettingsList.append(mssqlsettings.DistributedTransactionTraceValue("trace_log",
        "MSSQL_DTC_TRACE_LOG",
        mssqlsettings.SettingValueType.integer,
        _("Log tracing"),
        mssqlsettings.SectionForSetting.distributedtransaction,
        True))

    supportedSettingsList.append(mssqlsettings.TcpPortSetting("servertcpport",
        "MSSQL_DTC_TCP_PORT",
        mssqlsettings.SettingValueType.integer,
        _("MSDTC rpc server port"),
        mssqlsettings.SectionForSetting.distributedtransaction,
        True))

    supportedSettingsList.append(mssqlsettings.DirectorySetting("tracefilepath",
        "MSSQL_DTC_TRACE_FILE_PATH",
        mssqlsettings.SettingValueType.dirPath,
        _("Folder in which trace files should be stored"),
        mssqlsettings.SectionForSetting.distributedtransaction,
        True))

    supportedSettingsList.append(mssqlsettings.BooleanSetting("outboundnetworkaccess",
        None,
        mssqlsettings.SettingValueType.boolean,
        _("Enable outbound network access for sp_execute_external_script"),
        mssqlsettings.SectionForSetting.extensibility,
        False, # restart sqlserver not required
        "1", # true value
        "0")) # false value

    supportedSettingsList.append(mssqlsettings.DirectorySettingList("datadirectories",
        None,
        mssqlsettings.SettingValueType.string,
        _("Colon separated directory paths available to sp_execute_external_script"),
        mssqlsettings.SectionForSetting.extensibility,
        False, # restart sqlserver not required
        False, # hidden
        False)) # sectionOnly
    
    supportedSettingsList.append(mssqlsettings.RBinPathFileSetting("rbinpath",
        None,
        mssqlsettings.SettingValueType.filePath,
        _("Full path to R executable"),
        mssqlsettings.SectionForSetting.extensibility,
        False, # restart sqlserver not required
        False, # hidden
        False)) # sectionOnly

    supportedSettingsList.append(mssqlsettings.PythonBinPathFileSetting("pythonbinpath",
        None,
        mssqlsettings.SettingValueType.filePath,
        _("Full path to Python executable"),
        mssqlsettings.SectionForSetting.extensibility,
        False, # restart sqlserver not required
        False, # hidden
        False)) # sectionOnly

    supportedSettingsList.append(mssqlsettings.BooleanSetting("stoponguestprocessfault",
        "MSSQL_STOP_ON_GUEST_PROCESS_FAULT",
        mssqlsettings.SettingValueType.boolean,
        _("Stops the process if any guest process reports unhandled exception"),
        mssqlsettings.SectionForSetting.control,
        True, # restart sqlserver required
        "true", # true value
        "false")) # false value

    supportedSettingsList.append(mssqlsettings.GuidSetting("aadserveradminsid",
        "MSSQL_AAD_SERVER_ADMIN_SID",
        mssqlsettings.SettingValueType.string,
        _("SID of the Microsoft Entra ID account which will be made sysadmin"),
        mssqlsettings.SectionForSetting.network,
        True)) # restart sqlserver required

    supportedSettingsList.append(mssqlsettings.AADAdminNameSetting("aadserveradminname",
        "MSSQL_AAD_SERVER_ADMIN_NAME",
        mssqlsettings.SettingValueType.string,
        _("Name of the Microsoft Entra ID account which will be made sysadmin"),
        mssqlsettings.SectionForSetting.network,
        True)) # restart sqlserver required

    supportedSettingsList.append(mssqlsettings.AADAdminTypeSetting("aadserveradmintype",
        "MSSQL_AAD_SERVER_ADMIN_TYPE",
        mssqlsettings.SettingValueType.string,
        _("Type of the Microsoft Entra ID account which will be made sysadmin"),
        mssqlsettings.SectionForSetting.network,
        True)) # restart sqlserver required

    supportedSettingsList.append(mssqlsettings.AadCertificateSetting("aadcertificatefilepath",
        "MSSQL_AAD_CERTIFICATE_FILE_PATH",
        mssqlsettings.SettingValueType.string,
        _("Path to certificate file for authenticating to Microsoft Entra ID"),
        mssqlsettings.SectionForSetting.network,
        True)) # restart sqlserver required

    supportedSettingsList.append(mssqlsettings.GuidSetting("aadclientid",
        "MSSQL_AAD_CLIENT_ID",
        mssqlsettings.SettingValueType.string,
        _("Microsoft Entra ID Client GUID"),
        mssqlsettings.SectionForSetting.network,
        True)) # restart sqlserver required

    supportedSettingsList.append(mssqlsettings.GuidSetting("aadmsiclientid",
        "MSSQL_AAD_MSI_CLIENT_ID",
        mssqlsettings.SettingValueType.string,
        _("Microsoft Entra ID MSI Client GUID"),
        mssqlsettings.SectionForSetting.network,
        True)) # restart sqlserver required

    supportedSettingsList.append(mssqlsettings.GuidSetting("aadprimarytenant",
        "MSSQL_AAD_PRIMARY_TENANT",
        mssqlsettings.SettingValueType.string,
        _("Microsoft Entra ID Primary Tenant GUID"),
        mssqlsettings.SectionForSetting.network,
        True)) # restart sqlserver required

    supportedSettingsList.append(mssqlsettings.AadEndpointSetting("aadgraphapiendpoint",
        "MSSQL_AAD_GRAPH_API_ENDPOINT",
        mssqlsettings.SettingValueType.string,
        _("Endpoint for Microsoft Entra ID Graph API"),
        mssqlsettings.SectionForSetting.network,
        True)) # restart sqlserver required

    supportedSettingsList.append(mssqlsettings.AadEndpointSetting("aadfederationmetadataendpoint",
        "MSSQL_AAD_FEDERATION_METADATA_ENDPOINT",
        mssqlsettings.SettingValueType.string,
        _("Endpoint for Microsoft Entra ID Federation Metadata"),
        mssqlsettings.SectionForSetting.network,
        True)) # restart sqlserver required

    supportedSettingsList.append(mssqlsettings.AadEndpointSetting("aadauthenticationendpoint",
        "MSSQL_AAD_AUTHENTICATION_ENDPOINT",
        mssqlsettings.SettingValueType.string,
        _("Endpoint for Microsoft Entra ID Authentication"),
        mssqlsettings.SectionForSetting.network,
        True)) # restart sqlserver required

    supportedSettingsList.append(mssqlsettings.AadEndpointSetting("aadissuerurl",
        "MSSQL_AAD_ISSUER_URL",
        mssqlsettings.SettingValueType.string,
        _("Microsoft Entra ID Issuer URL"),
        mssqlsettings.SectionForSetting.network,
        True)) # restart sqlserver required

    supportedSettingsList.append(mssqlsettings.AadEndpointSetting("aadonbehalfofauthority",
        "MSSQL_AAD_ON_BEHALF_OF_AUTHORITY",
        mssqlsettings.SettingValueType.string,
        _("Microsoft Entra ID On Behalf of Authority"),
        mssqlsettings.SectionForSetting.network,
        True)) # restart sqlserver required

    supportedSettingsList.append(mssqlsettings.AadEndpointSetting("aadgraphendpoint",
        "MSSQL_AAD_MS_GRAPH_END_POINT",
        mssqlsettings.SettingValueType.string,
        _("Microsoft Entra ID Graph Endpoint"),
        mssqlsettings.SectionForSetting.network,
        True)) # restart sqlserver required

    supportedSettingsList.append(mssqlsettings.AadEndpointSetting("aadstsurl",
        "MSSQL_AAD_STS_URL",
        mssqlsettings.SettingValueType.string,
        _("Microsoft Entra ID STS URL"),
        mssqlsettings.SectionForSetting.network,
        True)) # restart sqlserver required

    supportedSettingsList.append(mssqlsettings.AadEndpointSetting("aadmsgraphendpoint",
        "MSSQL_AAD_MS_GRAPH_END_POINT",
        mssqlsettings.SettingValueType.string,
        _("Microsoft Entra ID MS Graph Endpoint"),
        mssqlsettings.SectionForSetting.network,
        True)) # restart sqlserver required

    supportedSettingsList.append(mssqlsettings.AadEndpointSetting("aadclientcertblacklist",
        "MSSQL_AAD_CLIENT_CERT_BLACKLIST",
        mssqlsettings.SettingValueType.string,
        _("Microsoft Entra ID Client Cert Blacklist"),
        mssqlsettings.SectionForSetting.network,
        True)) # restart sqlserver required

    supportedSettingsList.append(mssqlsettings.BooleanSetting("aadsendx5c",
        "MSSQL_AAD_SEND_X5C",
        mssqlsettings.SettingValueType.string,
        _("Microsoft Entra ID Send X5C"),
        mssqlsettings.SectionForSetting.network,
        True)) # restart sqlserver required

    supportedSettingsList.append(mssqlsettings.AadEndpointSetting("aadserviceprincipalname",
        "MSSQL_AAD_SERVICE_PRINCIPAL_NAME",
        mssqlsettings.SettingValueType.string,
        _("Microsoft Entra ID Service Principal Name"),
        mssqlsettings.SectionForSetting.network,
        True)) # restart sqlserver required

    supportedSettingsList.append(mssqlsettings.AadEndpointSetting("aadserviceprincipalnamenoslash",
        "MSSQL_AAD_SERVICE_PRINCIPAL_NAME_NO_SLASH",
        mssqlsettings.SettingValueType.string,
        _("Microsoft Entra ID Service Principal Name no slash"),
        mssqlsettings.SectionForSetting.network,
        True)) # restart sqlserver required

    supportedSettingsList.append(mssqlsettings.BooleanSetting("enabled",
        "MSSQL_WMI_ENABLED",
        mssqlsettings.SettingValueType.boolean,
        _("Enable or disable Wmi services"),
        mssqlsettings.SectionForSetting.wmi,
        True,
        "true",  # true value
        "false")) # false value

    supportedSettingsList.append(mssqlsettings.BooleanSetting("recheck",
        "MSSQL_WMI_RECHECK",
        mssqlsettings.SettingValueType.boolean,
        _("Recheck Wmi settings periodically"),
        mssqlsettings.SectionForSetting.wmi,
        True,
        "true",  # true value
        "false")) # false value

    supportedSettingsList.append(mssqlsettings.BooleanSetting("azurebilling",
        "MSSQL_AZURE_BILLING",
        mssqlsettings.SettingValueType.boolean,
        _("Use Azure billing for SQL Server activation instead of a retail product key or volume licensing."),
        mssqlsettings.SectionForSetting.licensing,
        False,
        "true",  # true value
        "false")) # false value

    supportedSettingsList.append(mssqlsettings.BooleanSetting("productcoveredbysa",
        "MSSQL_PRODUCT_COVERED_BY_SA",
        mssqlsettings.SettingValueType.boolean,
        _("The license used for this instance of SQL Server is covered by Software Assurance."),
        mssqlsettings.SectionForSetting.licensing,
        False,
        "yes",  # true value
        "no")) # false value

    supportedSectionsList = [value for key, value in list(mssqlsettings.SectionForSetting.__dict__.items()) if not key.startswith('__') and not callable(key)]


